#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "librmff.h"

char buffer[128000];

int show_frames = 0;
int calc_checksums = 0;
int dump_frames = 0;
int show_index = 0;
int show_headers = 0;

#define BASE 65521
#define A0 check += *adler_buffer++; sum2 += check;
#define A1 A0 A0
#define A2 A1 A1
#define A3 A2 A2
#define A4 A3 A3
#define A5 A4 A4
#define A6 A5 A5

uint32_t
calc_adler32(const unsigned char *adler_buffer,
             int size) {
  register uint32_t sum2, check;
  register int k;

  check = 1;
  k = size;

  sum2 = (check >> 16) & 0xffffL;
  check &= 0xffffL;
  while (k >= 64) {
    A6;
    k -= 64;
  }

  if (k)
    do {
      A0;
    } while (--k);

  check %= BASE;
  check |= (sum2 % BASE) << 16;

  return check;
}

#define C(c) ((c) >= 32) && ((c) <= 126) ? (c) : ' '

#define d4(l, a) dump4(l, (unsigned char *)&props->a)
#define d2(l, a) dump2(l, (unsigned char *)&props->a)

void
dump4(const char *l,
      unsigned char *c) {
  printf("  %20s: 0x%08x %10u '%c%c%c%c'\n", l, rmff_get_uint32_be(c),
         rmff_get_uint32_be(c), C(c[0]), C(c[1]), C(c[2]), C(c[3]));
}

void
dump2(const char *l,
      unsigned char *c) {
  printf("  %20s:     0x%04x %10u   '%c%c'\n", l, rmff_get_uint16_be(c),
         rmff_get_uint16_be(c), C(c[0]), C(c[1]));
}

void
dump_extra(unsigned char *p,
           uint32_t size) {
  int i;

  printf("  Extra: %u bytes: 0x", size);
  for (i = 0; i < size; i++)
    printf("%02x%c", p[i], ((i + 1) % 10) == 0 ? '|' :
           ((i + 1) % 2) == 0 ? '.' : ' ');
  printf("\n");
}

void
dump_audio_v4_props(rmff_track_t *track) {
  const real_audio_v4_props_t *props;
  uint32_t size;

  props = (real_audio_v4_props_t *)track->mdpr_header.type_specific_data;
  size = rmff_get_uint32_be(&track->mdpr_header.type_specific_size);
  printf("Dumping real_audio_v4_props_t for track %u:\n", track->id);
  d4("fourcc1", fourcc1);
  d2("version1", version1);
  d2("unknown1", unknown1);
  d4("fourcc2", fourcc2);
  d4("stream_length", stream_length);
  d2("version2", version2);
  d4("header_size", header_size);
  d2("flavor", flavor);
  d4("coded_frame_size", coded_frame_size);
  d4("unknown3", unknown3);
  d4("unknown4", unknown4);
  d4("unknown5", unknown5);
  d2("sub_packet_h", sub_packet_h);
  d2("frame_size", frame_size);
  d2("sub_packet_size", sub_packet_size);
  d2("unknown6", unknown6);
  d2("sample_rate", sample_rate);
  d2("unknown8", unknown8);
  d2("sample_size", sample_size);
  d2("channels", channels);
  if (size > sizeof(real_audio_v4_props_t)) {
    size -= sizeof(real_audio_v4_props_t);
    dump_extra((unsigned char *)(props + 1), size);
  }
}

void
dump_audio_v5_props(rmff_track_t *track) {
  const real_audio_v5_props_t *props;
  uint32_t size;

  props = (real_audio_v5_props_t *)track->mdpr_header.type_specific_data;
  size = rmff_get_uint32_be(&track->mdpr_header.type_specific_size);
  printf("Dumping real_audio_v5_props_t for track %u:\n", track->id);
  d4("fourcc1", fourcc1);
  d2("version1", version1);
  d2("unknown1", unknown1);
  d4("fourcc2", fourcc2);
  d4("stream_length", stream_length);
  d2("version2", version2);
  d4("header_size", header_size);
  d2("flavor", flavor);
  d4("coded_frame_size", coded_frame_size);
  d4("unknown3", unknown3);
  d4("unknown4", unknown4);
  d4("unknown5", unknown5);
  d2("sub_packet_h", sub_packet_h);
  d2("frame_size", frame_size);
  d2("sub_packet_size", sub_packet_size);
  d2("unknown6", unknown6);
  printf("%20s: 0x%02x %02x %02x %02x %02x %02x\n", "unknown7",
         props->unknown7[0], props->unknown7[1], props->unknown7[2],
         props->unknown7[3], props->unknown7[4], props->unknown7[5]);
  d2("sample_rate", sample_rate);
  d2("unknown8", unknown8);
  d2("sample_size", sample_size);
  d2("channels", channels);
  d4("genr", genr);
  d4("fourcc3", fourcc3);
  if (size > sizeof(real_audio_v5_props_t)) {
    size -= sizeof(real_audio_v5_props_t);
    dump_extra((unsigned char *)(props + 1), size);
  }
}

void
dump_video_props(rmff_track_t *track) {
  const real_video_props_t *props;
  uint32_t size;

  props = (real_video_props_t *)track->mdpr_header.type_specific_data;
  size = rmff_get_uint32_be(&track->mdpr_header.type_specific_size);
  printf("Dumping real_video_props_t for track %u:\n", track->id);
  d4("size", size);
  d4("fourcc1", fourcc1);
  d4("fourcc2", fourcc2);
  d2("width", width);
  d2("height", height);
  d2("bpp", bpp);
  d4("unknown1", unknown1);
  d4("fps", fps);
  d4("type1", type1);
  d4("type2", type2);
  if (size > sizeof(real_video_props_t)) {
    size -= sizeof(real_video_props_t);
    dump_extra((unsigned char *)(props + 1), size);
  }
}

void
dump_object_header(const rmff_object_t *props) {
  d4("object_id", id);
  d4("object_size", size);
  d2("object_version", version);
}

void
dump_rmf_header(rmff_file_t *file) {
  const rmff_rmf_t *props;

  props = &file->rmf_header;
  printf("Dumping RMF file header:\n");
  dump_object_header(&props->obj);
  d4("format_version", format_version);
  d4("num_headers", num_headers);
}

void
dump_prop_header(rmff_file_t *file) {
  const rmff_prop_t *props;

  props = &file->prop_header;
  printf("Dumping PROP file header:\n");
  dump_object_header(&props->obj);
  d4("max_bit_rate", max_bit_rate);
  d4("avg_bit_rate", avg_bit_rate);
  d4("max_packet_size", max_packet_size);
  d4("avg_packet_size", avg_packet_size);
  d4("num_packets", num_packets);
  d4("duration", duration);
  d4("preroll", preroll);
  d4("index_offset", index_offset);
  d4("data_offset", data_offset);
  d2("flags", flags);
}

void
dump_cont_header(rmff_file_t *file) {
  const rmff_cont_t *cont;

  if (!file->cont_header_present) {
    printf("File does not contain a CONT header.\n");
    return;
  }
  cont = &file->cont_header;
  printf("Dumping CONT file header:\n");
  dump_object_header(&cont->obj);
  printf("                 title: %s\n", cont->title != NULL ?
         cont->title : "");
  printf("                author: %s\n", cont->author != NULL ?
         cont->author : "");
  printf("             copyright: %s\n", cont->copyright != NULL ?
         cont->copyright : "");
  printf("               comment: %s\n", cont->comment != NULL ?
         cont->comment : "");
}

void
dump_mdpr_header(rmff_track_t *track) {
  const rmff_mdpr_t *props;

  props = &track->mdpr_header;
  printf("Dumping MDPR track header for track %u:\n", track->id);
  dump_object_header(&props->obj);
  d2("id", id);
  d4("max_bit_rate", max_bit_rate);
  d4("avg_bit_rate", avg_bit_rate);
  d4("max_packet_size", max_packet_size);
  d4("avg_packet_size", avg_packet_size);
  d4("start_time", start_time);
  d4("preroll", preroll);
  d4("duration", duration);
  printf("                  name: %s\n", props->name != NULL ?
         props->name : "");
  printf("             mime_type: %s\n", props->mime_type != NULL ?
         props->mime_type : "");
  d4("type_specific_size", type_specific_size);
}

void
dump_logical_fileinfo(rmff_track_t *track) {
  unsigned char *p, *end;
  uint32_t count, i, len;
  char *value;

  if (track->mdpr_header.type_specific_size < 12) {
    printf("Dumping the 'logical-fileinfo' failed: size (%u) too small.\n",
           track->mdpr_header.type_specific_size);
    return;
  }
  p = track->mdpr_header.type_specific_data;
  end = p + track->mdpr_header.type_specific_size;
  p += 8;
  count = rmff_get_uint32_be(p);
  p += 4;
  printf("Dumping %u entries from the 'logical-fileinfo' header.\n", count);
  for (i = 0; i < count; i++) {
    if ((p + 4) >= end)
      return;
    len = rmff_get_uint32_be(p);
    p += 4;
    if ((p + len) >= end)
      return;
    p++;
    len = rmff_get_uint16_be(p);
    p += 2;
    value = (char *)calloc(len + 1, 1);
    memcpy(value, p, len);
    p += len;
    p += 4;
    printf("  %s = ", value);
    free(value);
    len = rmff_get_uint16_be(p);
    p += 2;
    value = (char *)calloc(len + 1, 1);
    memcpy(value, p, len);
    p += len;
    printf("%s\n", value);
    free(value);
  }
}

void
test_reading(const char *file_name) {
  rmff_file_t *file;
  rmff_track_t *track;
  rmff_frame_t *frame;
  int frame_no, i, j;

  file = rmff_open_file(file_name, MB_OPEN_MODE_READING);
  if (file == NULL) {
    printf("Could not open %s\n", file_name);
    return;
  }
  printf("Opened %s, reading headers... ", file_name);
  if (rmff_read_headers(file) != RMFF_ERR_OK) {
    printf("failed. Error code: %d, error message: %s\n",
           rmff_last_error, rmff_last_error_msg);
    return;
  }
  printf("done.\nNumber of tracks: %u, number of packets: %u\nNow reading "
         "all frames.\n", file->num_tracks, file->num_packets_in_chunk);
  if (show_headers) {
    dump_rmf_header(file);
    dump_prop_header(file);
    dump_cont_header(file);
    for (i = 0; i < file->num_tracks; i++) {
      dump_mdpr_header(file->tracks[i]);
      if (file->tracks[i]->type == RMFF_TRACK_TYPE_AUDIO) {
        if (rmff_get_uint16_be(&((real_audio_v4_props_t *)
                                 &file->tracks[i]->mdpr_header)->version1) ==
            4)
          dump_audio_v4_props(file->tracks[i]);
        else
          dump_audio_v5_props(file->tracks[i]);
      } else if (file->tracks[i]->type == RMFF_TRACK_TYPE_VIDEO)
        dump_video_props(file->tracks[i]);
      else if (!strcmp(file->tracks[i]->mdpr_header.mime_type,
                       "logical-fileinfo"))
        dump_logical_fileinfo(file->tracks[i]);
    }
  }
  frame_no = 0;
  do {
    if (show_frames)
      printf("frame %6d expected size: %5d", frame_no,
             rmff_get_next_frame_size(file));
    if ((frame = rmff_read_next_frame(file, buffer)) != NULL) {
      if (show_frames) {
        printf(", read %5d for track %2u", frame->size, frame->id);
        if (calc_checksums)
          printf(" adler 0x%08x", calc_adler32(frame->data, frame->size));
      }
      if (dump_frames) {
        track = rmff_find_track_with_id(file, frame->id);
        if (track != NULL) {
          void *handle;
          char name[200];

          sprintf(name, "track-%02u-frame-%06u.bin", track->id,
                  (uint32_t)track->app_data);
          handle = std_mb_file_io.open(name, MB_OPEN_MODE_WRITING);
          if (handle != NULL) {
            std_mb_file_io.write(handle, frame->data, frame->size);
            std_mb_file_io.close(handle);
          }
          track->app_data = (void *)(((uint32_t)track->app_data) + 1);
        }
      }
      rmff_release_frame(frame);
    }
    if (show_frames)
      printf("\n");
    frame_no++;
  } while (frame != NULL);
  printf("Done reading frames.\n");
  if (show_index) {
    printf("Dumping indexes.\n");
    for (i = 0; i < file->num_tracks; i++) {
      track = file->tracks[i];
      if (track->num_index_entries > 0) {
        printf("Index for track %u: %u entries\n", track->id,
               track->num_index_entries);
        for (j = 0; j < track->num_index_entries; j++)
          printf("  %d: timecode: %u pos: %u packet_number: %u\n",
                 j, track->index[j].timecode, track->index[j].pos,
                 track->index[j].packet_number);
      }
    }
  }
  rmff_close_file(file);
}

int
main(int argc,
     char *argv[]) {
  int i;
  const char *input;

  input = "readtestvideo.rm";
  for (i = 1; i < argc; i++) {
    if (!strcmp(argv[i], "-i"))
      show_index = 1;
    else if (!strcmp(argv[i], "-f"))
      show_frames = 1;
    else if (!strcmp(argv[i], "-h"))
      show_headers = 1;
    else if (!strcmp(argv[i], "-d"))
      dump_frames = 1;
    else if (!strcmp(argv[i], "-c")) {
      show_frames = 1;
      calc_checksums = 1;
    } else
      input = argv[i];
  }

  test_reading(input);

  return 0;
}
