#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "librmff.h"

int
copy_this_track(uint16_t id,
                const char *selection) {
  char sid[7];
  int len;

  if (selection == NULL)
    return 1;

  sprintf(sid, "%u", id);
  selection += strspn(selection,",");
  while ((len = strcspn(selection,",")) > 0) {
    if (!strncmp(sid, selection, len))
      return 1;
    selection += len;
    selection += strspn(selection,",");
  }
  
  return 0;
  
}

#define tname(t) ((t->mdpr_header.name != NULL ? \
                  t->mdpr_header.name : "unnamed"))
#define tmime_type(t) ((t->mdpr_header.mime_type != NULL ? \
                        t->mdpr_header.mime_type : "no_mime_type"))

void
test_copying(const char *file_name,
             const char *track_selection) {
  rmff_file_t *src, *dst;
  rmff_track_t *tsrc, *tdst;
  unsigned char *buffer;
  rmff_frame_t *frame;
  int i, buffer_size, next_frame_size;

  src = rmff_open_file(file_name, MB_OPEN_MODE_READING);
  if (src == NULL) {
    printf("Could not open the source file '%s'\n", file_name);
    return;
  }
  dst = rmff_open_file("output.rm", MB_OPEN_MODE_WRITING);
  if (dst == NULL) {
    printf("Could not open the destination file 'output.rm'\n");
    return;
  }
  printf("Opened '%s' and 'output.rm', reading headers... ", file_name);
  if (rmff_read_headers(src) != RMFF_ERR_OK) {
    printf("failed. Error code: %d, error message: %s\n",
           rmff_last_error, rmff_last_error_msg);
    rmff_close_file(src);
    rmff_close_file(dst);
    return;
  }
  printf("done.\nNumber of tracks: %u, number of packets: %u\n",
         src->num_tracks, src->num_packets_in_chunk);
  dst->prop_header = src->prop_header;
  if (src->cont_header_present) {
    printf("CONT header present in source, copying.\n");
    dst->cont_header_present = 1;
    rmff_set_cont_header(dst, src->cont_header.title, src->cont_header.author,
                         src->cont_header.copyright, src->cont_header.comment);
  }

  for (i = 0; i < src->num_tracks; i++) {
    tsrc = src->tracks[i];
    if (!copy_this_track(tsrc->id, track_selection)) {
      printf("Skipping track %u (name: %s, MIME type: %s).\n", tsrc->id,
             tname(tsrc), tmime_type(tsrc));
      continue;
    }
    printf("Copying track %u (name: %s, MIME type: %s).\n", tsrc->id,
           tname(tsrc), tmime_type(tsrc));
    tdst = rmff_add_track(dst, tsrc->type != RMFF_TRACK_TYPE_UNKNOWN ? 1 : 0);
    if (tdst == NULL) {
      printf("Could not add a new track.\n");
      exit(1);
    }
    rmff_copy_track_headers(tdst, tsrc);
    tsrc->app_data = (void *)tdst->id;
  }
  printf("Writing headers... ");
  if (rmff_write_headers(dst) != RMFF_ERR_OK)
    printf("failed. Error code: %d, error message: %s\n",
           rmff_last_error, rmff_last_error_msg);
  else
    printf("done\n");

  buffer_size = 0;
  buffer = NULL;
  do {
    printf(".");
    next_frame_size = rmff_get_next_frame_size(src);
    if (next_frame_size <= 0)
      break;
    if (next_frame_size > buffer_size) {
      buffer = (unsigned char *)realloc(buffer, next_frame_size);
      buffer_size = next_frame_size;
    }
    frame = rmff_read_next_frame(src, buffer);
    if (frame == NULL)
      break;
    printf("r");
    tsrc = rmff_find_track_with_id(src, frame->id);
    if (tsrc == NULL) {
      printf("{ ");
      rmff_release_frame(frame);
      continue;
    }
    tdst = rmff_find_track_with_id(dst, (uint32_t)tsrc->app_data);
    if (tdst == NULL) {
      printf("} ");
      rmff_release_frame(frame);
      continue;
    }
    if (rmff_write_frame(tdst, frame) != RMFF_ERR_OK)
      printf("( ");
    else
      printf("w ");
  } while (next_frame_size > 0);

  printf("!\n");

  if (buffer != NULL)
    free(buffer);

  printf("Writing the index... ");
  if (rmff_write_index(dst) != RMFF_ERR_OK)
    printf("failed. Error code: %d, error message: %s\n",
           rmff_last_error, rmff_last_error_msg);
  else
    printf("done\n");

  printf("Fixing headers... ");
  if (rmff_fix_headers(dst) != RMFF_ERR_OK)
    printf("failed. Error code: %d, error message: %s\n",
           rmff_last_error, rmff_last_error_msg);
  else
    printf("done\n");

  rmff_close_file(src);
  rmff_close_file(dst);
}

int
main(int argc,
     char *argv[]) {
  if (argc < 2)
    test_copying("readtestvideo.rm", NULL);
  else {
    if (argc < 3)
      test_copying(argv[1], NULL);
    else
      test_copying(argv[1], argv[2]);
  }

  return 0;
}
