/*
  mkvmerge -- utility for splicing together matroska files
      from component media subtypes

  r_mp3.h

  Written by Moritz Bunkus <moritz@bunkus.org>

  Distributed under the GPL
  see the file COPYING for details
  or visit http://www.gnu.org/copyleft/gpl.html
*/

/*!
    \file
    \version $Id: r_wav.cpp 1434 2004-01-22 22:36:05Z mosu $
    \brief MP3 reader module
    \author Moritz Bunkus <moritz@bunkus.org>
    \author Peter Niemayer <niemayer@isg.de>
*/

#include "os.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#if defined(COMP_CYGWIN)
#include <sys/unistd.h>         // Needed for swab()
#elif __GNUC__ == 2
#define __USE_XOPEN
#include <unistd.h>
#else
#include <unistd.h>
#endif

#include "mkvmerge.h"
#include "common.h"
#include "error.h"
#include "r_wav.h"
#include "p_pcm.h"
#include "p_dts.h"
#include "dts_common.h"

extern "C" {
#include <avilib.h> // for wave_header
}

int
wav_reader_c::probe_file(mm_io_c *mm_io,
                         int64_t size) {
  wave_header wheader;

  if (size < sizeof(wave_header))
    return 0;
  try {
    mm_io->setFilePointer(0, seek_beginning);
    if (mm_io->read((char *)&wheader, sizeof(wheader)) != sizeof(wheader))
      return 0;
    mm_io->setFilePointer(0, seek_beginning);
  } catch (exception &ex) {
    return 0;
  }
  if (strncmp((char *)wheader.riff.id, "RIFF", 4) ||
      strncmp((char *)wheader.riff.wave_id, "WAVE", 4) ||
      strncmp((char *)wheader.data.id, "data", 4))
    return 0;

  return 1;
}

wav_reader_c::wav_reader_c(track_info_c *nti)
  throw (error_c):
  generic_reader_c(nti) {
  int64_t size;

  pcmpacketizer = 0;
  dtspacketizer = 0;

  try {
    mm_io = new mm_io_c(ti->fname, MODE_READ);
    mm_io->setFilePointer(0, seek_end);
    size = mm_io->getFilePointer();
    mm_io->setFilePointer(0, seek_beginning);
  } catch (exception &ex) {
    throw error_c("wav_reader: Could not open the source file.");
  }
  if (!wav_reader_c::probe_file(mm_io, size))
    throw error_c("wav_reader: Source is not a valid WAVE file.");
  if (mm_io->read(&wheader, sizeof(wheader)) != sizeof(wheader))
    throw error_c("wav_reader: could not read WAVE header.");
  bps = get_uint16(&wheader.common.wChannels) *
    get_uint16(&wheader.common.wBitsPerSample) *
    get_uint32(&wheader.common.dwSamplesPerSec) / 8;
  chunk = (unsigned char *)safemalloc(bps + 1);
  bytes_processed = 0;
  ti->id = 0;                   // ID for this track.

  {
    // check wether .wav file contains DTS data...
    unsigned short obuf[max_dts_packet_size/2];
    unsigned short buf[2][max_dts_packet_size/2];
    int cur_buf = 0;

    long rlen = mm_io->read(obuf, max_dts_packet_size);
    mm_io->setFilePointer(sizeof(wheader), seek_beginning);

    for (dts_swap_bytes = 0; dts_swap_bytes < 2; dts_swap_bytes++) {
      memcpy(buf[cur_buf], obuf, rlen);

      if (dts_swap_bytes) {
        swab((const char *)buf[cur_buf], (char *)buf[cur_buf^1], rlen);
        cur_buf ^= 1;
      }

      for (dts_14_16 = 0; dts_14_16 < 2; dts_14_16++) {
        long erlen = rlen;
        if (dts_14_16) {
          unsigned long words = rlen / (8*sizeof(short));
          dts_14_to_dts_16(buf[cur_buf], words*8, buf[cur_buf^1]);
          cur_buf ^= 1;
        }

        dts_header_t dtsheader;
        int pos = find_dts_header((const unsigned char *)buf[cur_buf], erlen,
                                  &dtsheader);

        if (pos >= 0) {
          if (verbose) {
            mxinfo("Using WAV demultiplexer for %s.\n"
                   "+-> Using DTS output module for audio stream. %s %s\n",
                   ti->fname, (dts_swap_bytes)? "(bytes swapped)" : "",
                   (dts_14_16)? "(DTS14 encoded)" : "(DTS16 encoded)");
            print_dts_header(&dtsheader);
            is_dts = true;
          }

          dtspacketizer = new dts_packetizer_c(this, dtsheader, ti);
          // .wav's with DTS are always filled up with other stuff to match
          // the bitrate...
          dtspacketizer->skipping_is_normal = true;
          break;
        }

      }

      if (dtspacketizer)
        break;
    }
  }

  if (!dtspacketizer) {
    pcmpacketizer =
      new pcm_packetizer_c(this, get_uint32(&wheader.common.dwSamplesPerSec),
                           get_uint16(&wheader.common.wChannels),
                           get_uint16(&wheader.common.wBitsPerSample), ti);

    if (verbose)
      mxinfo("Using WAV demultiplexer for %s.\n+-> Using "
             "PCM output module for audio stream.\n", ti->fname);
    is_dts = false;
  }
}

wav_reader_c::~wav_reader_c() {
  delete mm_io;
  if (chunk != NULL)
    safefree(chunk);
  if (pcmpacketizer != NULL)
    delete pcmpacketizer;
  if (dtspacketizer != NULL)
    delete dtspacketizer;
}

int
wav_reader_c::read(generic_packetizer_c *) {
  if (pcmpacketizer) {
    int nread;

    nread = mm_io->read(chunk, bps);
    if (nread <= 0) {
      pcmpacketizer->flush();
      return 0;
    }

    pcmpacketizer->process(chunk, nread);

    bytes_processed += nread;

    if (nread != bps) {
      pcmpacketizer->flush();
      return 0;
    } else
      return EMOREDATA;
  }

  if (dtspacketizer) {
    unsigned short buf[2][max_dts_packet_size/2];
    int cur_buf = 0;
    long rlen = mm_io->read(buf[cur_buf], max_dts_packet_size);

    if (rlen <= 0) {
      dtspacketizer->flush();
      return 0;
    }

    if (dts_swap_bytes) {
      swab((const char *)buf[cur_buf], (char *)buf[cur_buf^1], rlen);
      cur_buf ^= 1;
    }

    long erlen = rlen;
    if (dts_14_16) {
      unsigned long words = rlen / (8*sizeof(short));
      //if (words*8*sizeof(short) != rlen) {
      // unaligned problem, should not happen...
      //}
      dts_14_to_dts_16(buf[cur_buf], words*8, buf[cur_buf^1]);
      cur_buf ^= 1;
      erlen = words * 7 * sizeof(short);
    }

    dtspacketizer->process((unsigned char *) (buf[cur_buf]), erlen);

    bytes_processed += rlen;

    if (rlen != max_dts_packet_size) {
      dtspacketizer->flush();
      return 0;
    } else
      return EMOREDATA;
  }

  return 0;
}

int
wav_reader_c::display_priority() {
  return DISPLAYPRIORITY_HIGH - 1;
}

void
wav_reader_c::display_progress(bool final) {
  int samples = (get_uint32(&wheader.riff.len) - sizeof(wheader) + 8) / bps;

  if (final)
    mxinfo("progress: %d/%d seconds (100%%)\r", (int)samples, (int)samples);
  else
    mxinfo("progress: %d/%d seconds (%d%%)\r",
           (int)(bytes_processed / bps), (int)samples,
           (int)(bytes_processed * 100L / bps / samples));
}

void
wav_reader_c::set_headers() {
  if (pcmpacketizer)
    pcmpacketizer->set_headers();
  if (dtspacketizer)
    dtspacketizer->set_headers();
}

void
wav_reader_c::identify() {
  mxinfo("File '%s': container: WAV\nTrack ID 0: audio (%s)\n",
         ti->fname, is_dts ? "DTS" : "PCM");
}
